/*
 *  bitflip.c
 *
 *  Copyright (C) 2007 Wim Decroix
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */
#define _GNU_SOURCE

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>

#define PROGRAM "bitflip"
#define VERSION "0.4"


/* For TV550, NXP uses for 1 page the following layout: (512 Data + 16 ECC ) * 4 . The tool expects 2048 + 64. Changing the tool to small page would resolve this.*/
/* #define MAINSIZE 2048
#define SPARESIZE 64 */
#define MAINSIZE 512
#define SPARESIZE 16

#define PAGESIZE (MAINSIZE+SPARESIZE)
#define BITSINBYTE 8

/* For TV550, NXP uses for 1 page the following layout: (512 Data + 16 ECC ) * 4 . The tool expects 2048 + 64. Changing the tool to small page would resolve this.*/
/* OOB settings */
/* #define PAGEDATAAREAS   4 */
/* #define SPAREECCBYTEOFFSET  6 */   /* not needed */
/* #define SPAREECCBYTELENGTH  1 */   /* not needed */
/* #define MAINECCBYTEOFFSET   7 */
/* #define MAINECCBYTELENGTH   4 */
#define PAGEDATAAREAS   1
#define SPAREECCBYTEOFFSET  0   /* not needed */
#define SPAREECCBYTELENGTH  16   /* not needed */
#define MAINECCBYTEOFFSET   0
#define MAINECCBYTELENGTH   0

static int count = 1;
static int max = 0;
static int mainarea = 0;
static int sparearea = 0;
static int skipemptypages = 0;
static char* inputfile = NULL;
static char* outputfile = NULL;

void display_help (void)
{
    printf("Usage: " PROGRAM " [OPTION] INPUTFILE OUTPUTFILE\n"
            "Flips bits in a flash dump image.  The flash dump needs to contain ECC data.\n"
            "Default: 1 bitflip in complete image (main or spare area)\n"
            "\n"
            "  -1, --one            One random bitflip in the complete image.\n"
            "  -c, --count <nr>     <Nr> random bitflip(s) in the complete image.\n"
            "  -a, --max            As much bitflips as possible.\n"
            "  -e, --skipemptypages Skip bitflips in empty pages (in combination with max). \n"
            "  -s, --spare          Only bitflip in spare area. \n"
            "  -m, --main           Only bitflip in main area. \n"
            "  -r, --seed <seed>    Seed for random generator. \n"
            "      --help           Display this help and exit.\n"
            "      --version        Output version information and exit.\n");
    exit(0);
}

void display_version (void)
{
    printf(PROGRAM " " VERSION "\n"
            "\n"
            "Copyright (C) 2009 Wim Decroix \n"
            "\n"
            PROGRAM " comes with NO WARRANTY\n"
            "to the extent permitted by law.\n"
            "\n"
            "You may redistribute copies of " PROGRAM "\n"
            "under the terms of the GNU General Public Licence.\n"
            "See the file `COPYING' for more information.\n");
    exit(0);
}

void process_options (int argc, char *argv[])
{
    int error = 0;
    
    for (;;) {
        int option_index = 0;
        static const char *short_options = "1c:asemr:";
        static const struct option long_options[] = {
            {"help", no_argument, 0, 0},
            {"version", no_argument, 0, 0},
            {"one", no_argument, 0, '1'},
            {"count", required_argument, 0, 'c'},
            {"max", no_argument, 0, 'a'},
            {"spare", no_argument, 0, 's'},
            {"main", no_argument, 0, 'm'},
            {"seed", required_argument, 0, 'r'},
            {"skipemptypages", no_argument, 0, 'e'},
            {0, 0, 0, 0},
        };

        int c = getopt_long(argc, argv, short_options,
                long_options, &option_index);
        if (c == EOF) {
            break;
        }

        switch (c) {
            case 0:
                switch (option_index) {
                    case 0:
                        display_help();
                        break;
                    case 1:
                        display_version();
                        break;
                }
                break;
            case '1':
                count = 1;
                break;
            case 'c':
                count = atoi(optarg);
                break;
            case 'a':
                max = 1;
                break;
            case 's':
                sparearea = 1;
                break;
            case 'm':
                mainarea = 1;
                break;
            case 'r':
                srand(atoi(optarg));
                break;
            case 'e':
                skipemptypages = 1;
                break;
            case '?':
            default:
                error = 1;
                break;
        }
    }
    
    if ((argc - optind) != 2 || error)
        display_help ();

    if( skipemptypages && !max )
        display_help ();

    inputfile = argv[optind++];
    outputfile = argv[optind++];
}

int myrand(int top)
{
    /*Random algorithm*/
    return (int) (((float)top) * (rand() / (RAND_MAX + 1.0))); 
    
    /*page + 1 incrementing algorithm*/
    /* static int counter=0; */
    /* int ret = (( counter + ( PAGESIZE * BITSINBYTE * counter )) % top ); */
    /* counter++; */
    /* return (int) ret; */
    
    /*increment algorithm*/
    /* static int counter=0; */
    /* return (int) ( counter++ % top); */
}

int emptypage( unsigned char* buff, int offset, int len )
{
    unsigned char* pageptr = buff + offset;
    int pageempty = 1;
    do {
        if (*pageptr++ != 0xff) {
            pageempty = 0;
            break;
        }
    }
    while (--len);
    return pageempty;
}

void bitflip( void )
{
    int infd, outfd;
    int filesize;
    struct stat statbuf;
    unsigned char* inbuf;
    unsigned char* outbuf;
    
    if ((infd = open(inputfile, O_RDONLY)) == -1)
    {
        perror("Error opening input file.");
        exit(1);
    }
    
    if (fstat(infd,&statbuf))
    {
        perror("Error stat file.");
        exit(1);
    }
    filesize = statbuf.st_size;

    if ( (filesize % PAGESIZE) != 0)
    {
        perror("Invalid input file (filesize not matched to PAGESIZE - The flash dump needs to contain OOB data).");
        close(infd);
        exit(1);
    }
    
    if ((outfd = open(outputfile, O_RDWR | O_CREAT, 0666)) == -1)
    {
        perror("Error opening output file.");
        exit(1);
    }
    
    if (ftruncate(outfd, filesize))
    {
        perror("Error truncating output file.");
        exit(1);
    }
    
    inbuf = mmap(0, filesize, PROT_READ, MAP_SHARED, infd, 0);
    if (inbuf == (void*)MAP_FAILED)
    {
        perror("Error mmap input file.");
        exit(1);
    }
    outbuf = mmap(0, filesize, PROT_READ|PROT_WRITE, MAP_SHARED, outfd, 0);
    if (outbuf == (void*)MAP_FAILED)
    {
        perror("Error mmap input file.");
        exit(1);
    }
    
    memcpy(outbuf, inbuf, filesize);
    
    if ((!sparearea) && (!mainarea))
    {
        /* make  bitflips by default in the main area */
        sparearea = 0;
        mainarea = 1;
    }
    
    if (!max)
    {
        int i;
        int nrbits = 0;
        if (sparearea) nrbits +=  (filesize / PAGESIZE) * BITSINBYTE * SPARESIZE; 
        if (mainarea) nrbits += (filesize / PAGESIZE) * BITSINBYTE * MAINSIZE;
        
        for( i = 0; i < count ; i++)
        {
            int offset = myrand(nrbits);    //number from 0->nrbits (returns a bitlocation in the file)
            int byte = offset >> 3;         //convert bits to bytes
            int bit = 1 << (offset & 7);    //random bit of the byte to invert depends on the random offset
            int byteoffset = 0;
            if (sparearea && mainarea) byteoffset = byte;
            else if (mainarea) byteoffset = (byte / MAINSIZE) * PAGESIZE + (byte & (MAINSIZE-1)) ;
            else if (sparearea) byteoffset = (byte / SPARESIZE) * PAGESIZE + (byte & (SPARESIZE-1)) + MAINSIZE ;
            outbuf[byteoffset] = inbuf[byteoffset] ^ bit;
            printf("0x%08X 0x%02X 0x%02X 0x%02X\n", byteoffset, inbuf[byteoffset], outbuf[byteoffset], bit);
        }
    }
    else
    {
        /* Max: in every page an error in the free OOB bytes (ECC of data in OOB) and an error per free data block (in ECC or data) */
        int i;
        int offset;
        int byte;
        int bit;
        int byteoffset = 0;
        for( i = 0; i< (filesize/PAGESIZE) ; i++ )
        {
            int skipemptypage = skipemptypages? emptypage( outbuf, i*PAGESIZE, PAGESIZE) : 0 ;
            if ( skipemptypage )
                continue;
            if (mainarea)
            {
                int j;
                for (j=0 ; j< PAGEDATAAREAS ; j++)
                {
                    int nrbits = ((MAINSIZE / PAGEDATAAREAS) + MAINECCBYTELENGTH) * BITSINBYTE;
                    offset = myrand(nrbits);
                    byte = offset >> 3;
                    bit = 1 << (offset & 7);
                    if(byte >= (MAINSIZE / PAGEDATAAREAS))
                    {/* make error in ECC of the data area */
                        byteoffset = (byte - (MAINSIZE / PAGEDATAAREAS))/* give the byteoffset of the spare area only */
                                   + (i*PAGESIZE)                       /* go to the right page */
                                   + MAINSIZE                           /* point to spare area */
                                   + ((SPARESIZE/PAGEDATAAREAS) * j )   /* go to the right subspace in the spare area */
                                   + MAINECCBYTEOFFSET;                 /* go to the right spare area DataECC offset*/
                    }
                    else
                    {/* make error in data area */
                        byteoffset = byte+ (i*PAGESIZE) + (j*(MAINSIZE / PAGEDATAAREAS));
                    }
                    outbuf[byteoffset] = inbuf[byteoffset] ^ bit;
                    printf("0x%08X 0x%02X 0x%02X 0x%02X\n", byteoffset, inbuf[byteoffset], outbuf[byteoffset], bit);
                }
            }
            if (sparearea)
            {
                int j;
                for (j=0 ; j< PAGEDATAAREAS ; j++)
                {
                    int nrbits = ((SPARESIZE / PAGEDATAAREAS) - (MAINECCBYTELENGTH)) * BITSINBYTE;
                    offset = myrand(nrbits);
                    byte = offset >> 3;
                    bit = 1 << (offset & 7);
                    byteoffset = byte+ (i*PAGESIZE) + MAINSIZE +(j*(SPARESIZE / PAGEDATAAREAS)) + ( (byte >= MAINECCBYTEOFFSET) ? MAINECCBYTELENGTH : 0 );
                    outbuf[byteoffset] = inbuf[byteoffset] ^ bit;
                    printf("0x%08X 0x%02X 0x%02X 0x%02X\n", byteoffset, inbuf[byteoffset], outbuf[byteoffset], bit);
                }
            }
        }
    }
    msync(outbuf, filesize, MS_SYNC);
    munmap(outbuf, filesize);
    munmap(inbuf, filesize);
    fsync(outfd);
    close(infd);
    close(outfd);
}

int main(int argc, char* argv[])
{
    process_options(argc, argv);
    bitflip();
    printf("Done..\n");
    return 0;
}
